<?php

namespace Modules\AiTriage\Listeners;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

class NewConversationListener
{
    /**
     * Handle the conversation.created event.
     * 
     * Sends conversation data to Replit webhook for AI analysis.
     *
     * @param \App\Conversation $conversation
     * @return void
     */
    public function handleNewConversation($conversation)
    {
        try {
            $webhookUrl = env('REPLIT_WEBHOOK_URL', 'https://config.clearcafe.com/webhook');
            
            // Get the first thread (original email message)
            $firstThread = $conversation->threads()->first();
            
            if (!$firstThread) {
                \Log::warning("AI Triage: No threads found for conversation {$conversation->id}");
                return;
            }
            
            // Prepare webhook payload
            $payload = [
                'conversation_id' => $conversation->id,
                'subject' => $conversation->subject,
                'body' => strip_tags($firstThread->body),  // Remove HTML tags
                'from' => $conversation->customer ? $conversation->customer->email : 'unknown@example.com',
            ];
            
            // Generate HMAC signature for webhook authentication
            $jsonPayload = json_encode($payload);
            $webhookSecret = env('WEBHOOK_SECRET', '');
            $signature = hash_hmac('sha256', $jsonPayload, $webhookSecret);
            
            // Send POST request to Replit webhook
            $client = new Client([
                'timeout' => 10.0,
                'verify' => true,  // Verify SSL certificates
            ]);
            
            $response = $client->post($webhookUrl, [
                'json' => $payload,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'X-Webhook-Signature' => $signature,
                ]
            ]);
            
            if ($response->getStatusCode() === 202 || $response->getStatusCode() === 200) {
                \Log::info("AI Triage: Webhook accepted for conversation {$conversation->id}");
            } else {
                \Log::warning("AI Triage: Unexpected response code {$response->getStatusCode()} for conversation {$conversation->id}");
            }
            
        } catch (RequestException $e) {
            \Log::error("AI Triage: HTTP error for conversation {$conversation->id}: " . $e->getMessage());
        } catch (\Exception $e) {
            \Log::error("AI Triage: Error processing conversation {$conversation->id}: " . $e->getMessage());
        }
    }
}
